from abc import ABC, abstractmethod
from typing import List, Dict, Any, Optional, Tuple

class BaseImportanceAnalyzer(ABC):
    def __init__(self):
        """
        Base class for all hyperparameter importance analyzers.
        Defines the common interface.
        """
        self.feature_importances_: Optional[Dict[str, float]] = None
        self.interaction_importances_: Optional[Dict[Tuple[str, str], float]] = None
        self.mode: Optional[str] = 'max'  # Optimization mode, e.g., 'min' or 'max'

    @abstractmethod
    def fit(self, configs: List[Dict[str, Any]], scores: List[float]) -> None:
        """
        Fit the analyzer based on hyperparameter configurations and their scores.
        """
        pass

    @abstractmethod
    def explain(self) -> Dict[str, float]:
        """
        Get the raw importance scores for each hyperparameter individually.

        Returns:
            A dictionary mapping hyperparameter names to their importance values.
        """
        pass

    @abstractmethod
    def explain_interactions(self) -> Dict[Tuple[str, str], float]:
        """
        Get the importance scores for hyperparameter pairs (interactions).

        Returns:
            A dictionary mapping (hyperparameter1, hyperparameter2) tuples to their joint importance values.
        """
        pass

    @abstractmethod
    def rank(self) -> List[str]:
        """
        Get a sorted list of hyperparameters by their individual importance (most important first).

        Returns:
            A list of hyperparameter names, sorted by importance.
        """
        pass
